import numpy as np
import scipy.signal as sig
import scipy.fft as ft

def create_time_data(num, samp_freq):
    
    delta_t = 1./samp_freq
    start, stop = 0, (num-1)*delta_t

    t = np.linspace(start, stop, num)

    return t


def create_waveform_data(t, a, f, shape):

    num = len(t)

    if shape == 'Sine':
        y = a*np.sin(2*np.pi*f*t)
    elif shape == 'Cosine':
        y = a*np.cos(2*np.pi*f*t)
    elif shape == 'DC Level':
        y = a*np.ones(num)
    elif shape == 'Square':
        y = a*sig.square(2*np.pi*f*t)
    elif shape == 'Sawtooth':
        y = a*sig.sawtooth(2*np.pi*f*t, 1.0)
    elif shape == 'Triangle':
        y = a*sig.sawtooth(2*np.pi*f*t, 0.5)
    elif shape == 'User-Defined':
        y = np.sin(2*np.pi*250*t)
        y = 10*y + 1*y**2
#       y = 10*y + 1*y**2 + 0.5*y**3
        
    return y


def sum_sines(t, num_terms):

    num = len(t)
    y = np.zeros(num)
    largest_odd = 2*num_terms + 1
    

    for odd in range(1, largest_odd, 2):
       y += (1/odd)*np.sin(2*np.pi*odd*t)
    y = (4/np.pi)*y

    return y


def projectile(thetas, v, h):

    g = 9.8
    ranges = []
    for theta in thetas:

        theta_rad = theta*np.pi/180
        x = np.sin(2*theta_rad)
        y = np.cos(theta_rad)

        alpha = v**2/(2*g)
        beta = 8*g*h/v**2
   
        r = alpha*(x + np.sqrt(x**2 + beta*y**2))
        ranges.append(r)

        max_index = np.argmax(ranges)
        max_angle = thetas[max_index]

    return ranges, max_angle


def falling_in_air(num,delta_t,alpha):\
    
    v = np.zeros(num)
    g = 9.8
    for i in range(2,num):
        v[i] = v[i-1] + (g-alpha*v[i-1]**2)*delta_t

    return v


def fft_mag_only(y, num, samp_freq):

    delta_t = 1/samp_freq
    n = num//2

    freq = ft.fftfreq(num, delta_t)

    fourier_transform = ft.fft(y)
    amp = (2/num)*np.abs(fourier_transform)
    amp[0] = amp[0]/2

    freq_positive = freq[0:n]
    amp_positive = amp[0:n]

    return freq_positive, amp_positive


def create_window(win_type, num):

    if win_type == 'Rectangle':
        win_type = 'boxcar'
    else:
        win_type = win_type.lower()

    win = sig.windows.get_window(win_type, num)

    coherent_gain = np.mean(win)
    win = win/coherent_gain

    return win


def fft_complex(y, num, samp_freq):

    delta_t = 1/samp_freq

    freq = ft.fftfreq(num, delta_t)
    complex_amp = (1/num)*ft.fft(y)

    return freq, complex_amp


def estimated_freq_amp(freq_positive, amp_positive, num,
                       win):

    f = freq_positive
    a = amp_positive

    freq_est = np.sum(f*a**2)/np.sum(a**2)

    enbw = num*np.sum(win**2)/np.sum(win)**2
    amp_est = np.sqrt(np.sum(a**2)/enbw)

    return freq_est, amp_est


if __name__ == '__main__':

    import matplotlib.pyplot as plt

    num, samp_freq = 100, 100
    a, f = 5, 3
    shape = 'Sine'
    
    t = create_time_data(num, samp_freq)
    y = create_waveform_data(t, a, f, shape)
    
    plt.plot(t, y)
    plt.show()
